#!/usr/bin/python
"""
    DrawingDIFF PDF/Image Compare
    
    Copyright (C) <2021-2026> <Scorch>              
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Version 0.01
    - Initial code

    Version 0.02
    - Updated to use Ghostscript in the distribution folder

    Version 0.03
    - Fixed color selectors in the general settings
    - Colors and Display Option are now saved in the settings file when the save button is clicked

    Version 0.04
    - Updated to be compatable with Python 3.14 and later
    - Switched PageUp PageDown keyboard bindings to be consistent with standard conventions
    - Fixed issue with number of pages nut updating properly when loading new files
    - Added command line options

"""
version = '0.04'
title_text = "DrawingDIFF V"+version
QUIET = False
DEBUG = False

import sys
VERSION = sys.version_info[0]

if VERSION == 3:
    from tkinter import *
    from tkinter.filedialog import *
    import tkinter.messagebox
    MAXINT = sys.maxsize
    from tkinter import ttk
    from tkinter.colorchooser import askcolor
    def trace_variable(variable, callback):
        return variable.trace_add("write", callback)
    def trace_delete(variable,callback):
        return variable.trace_remove("write",callback)  
    
else:
    from Tkinter import *
    from tkFileDialog import *
    import tkMessageBox
    MAXINT = sys.maxint
    import ttk
    from tkColorChooser import askcolor
    def trace_variable(variable, callback):
        return variable.trace_variable("w", callback)
    def trace_delete(variable,callback):
        return variable.trace_vdelete("w",callback)

if VERSION < 3 and sys.version_info[1] < 6:
    def next(item):
        return item.next()

import os
import getopt
import webbrowser
import struct
import tkinterdnd2 #https://github.com/Eliav2/tkinterdnd2

import math
import time
import re
import binascii
import operator
import traceback
import tempfile, shutil
import io
import warnings
import base64
import locale

from PIL import Image
from PIL import ImageTk
from PIL import ImageOps
from PIL import ImageSequence
from PIL import ImageMath
from PIL import ImageFilter    
Image.MAX_IMAGE_PIXELS = None # suppress DecompressionBombError

##try:
##    Image.LANCZOS
##except:
##    Image.LANCZOS=Image.LANCZOS

#try:
#    os.chdir(os.path.dirname(__file__))
#except:
#    pass

################################################################################
class Application(Frame):
    def __init__(self, master):
        Frame.__init__(self, master)
        self.w = 0
        self.h = 0
        frame = Frame(master)
        self.master = master
        self.x = -1
        self.y = -1            
        self.createWidgets()
        self.initComplete = 1
        self.menu_View_Refresh()

    def createWidgets(self):
        self.initComplete = 0
        self.batch = 0
        self.master.bind("<Configure>", self.Master_Configure)
        self.master.bind('<Enter>'    , self.bindConfigure)
        self.master.bind('<Escape>'   , self.KEY_ESC)
        self.master.bind('<F1>'       , self.KEY_F1)
        self.master.bind('<F2>'       , self.KEY_F2)
        self.master.bind('<F3>'       , self.KEY_F3)
        self.master.bind('<F4>'       , self.KEY_F4)
        self.master.bind('<F5>'       , self.KEY_F5)
        self.master.bind('<Prior>'    , self.page_down)
        self.master.bind('<Next>'     , self.page_up)

        self.highlight    = BooleanVar()
        self.show_grid    = BooleanVar()
        self.show_row     = BooleanVar()
        self.show_col     = BooleanVar()
        
        self.sheet_size   = StringVar()
        self.resolution   = StringVar()
        self.gs_path      = StringVar()
        self.current_input_file = StringVar()

        self.sheet_number = StringVar()
        self.image_display = StringVar()

        self.IMAGE_FILE_old   = StringVar()
        self.IMAGE_FILE_new   = StringVar()
        
        self.image_display.set("comb") #Options: old new comb
        self.sheet_number.set("0")
        ###########################################################################
        #                         INITILIZE VARIABLES                             #
        #    if you want to change a default setting this is the place to do it   #
        ###########################################################################
        self.show_grid.set(1)
        self.show_col.set(1)
        self.show_row.set(1)
        self.highlight.set(1)
        self.sheet_size.set("A")        # Options are "A","B","C","D","E","F"
        self.resolution.set("150")
        
        
        self.IMAGE_FILE_old.set(os.path.expanduser("~")+"/None")
        self.IMAGE_FILE_new.set(os.path.expanduser("~")+"/None")
        
        self.aspect_ratio =  0
        self.SCALE = 1
        
        self.settings = []
        self.segID = []

        # PAN and ZOOM STUFF
        self.panx = 0
        self.panx = 0
        self.lastx = 0
        self.lasty = 0
        self.making_shape = False
        self.shape_coords=[]

        self.HOME_DIR = os.path.expanduser("~")
        self.gs_path.set("gs9.26\\bin\\gswin64c.exe")
        self.Notation_boxes = []
        
        self.im  = []
        self.wim = 1
        self.him = 1
        self.aspect_ratio =  1


        self.ui_TKimage_old = []
        self.im_old  = [] 
        self.wim_old = 1 
        self.him_old = 1 
        self.aspect_ratio_old =  1

        self.ui_TKimage_new = []
        self.im_new  = [] 
        self.wim_new = 1 
        self.him_new = 1 
        self.aspect_ratio_new =  1 #float(self.wim_new-1) / float(self.him_new-1)
        
        self.statusMessage = StringVar()
        self.statusMessage.set("Welcome to DrawingDIFF")


        self.OLD_COLOR       = (0  ,0  ,255,255)
        self.NEW_COLOR       = (255,0  ,0  ,255)
        self.BOTH_COLOR      = (0  ,0  ,0  ,255)
        self.HIGHLIGHT_COLOR = (255,250,0  ,255)
            
        ##########################################################################
        ###                     END INITILIZING VARIABLES                      ###
        ##########################################################################

        ##########################################################################
        #                  Config File and command line options                  #
        ##########################################################################
        config_file = "DrawingDIFF.txt"
        home_config1 = os.path.expanduser("~") + "/" + config_file
        if ( os.path.isfile(config_file) ):
            self.Open_Settings_File(config_file)
        elif ( os.path.isfile(home_config1) ):
            self.Open_Settings_File(home_config1)

        opts, args = None, None
        try:
            opts, args = getopt.getopt(sys.argv[1:], "hb1:2:o:ts:",["help", "batch", "old", "new", "out", "tif", "settings"])
        except:
            fmessage('Unable interpret command line options')
            sys.exit()
        Load_Images=False
        make_TIF=False
        outputfile="default"
        for option, value in opts:
            if option in ('-h','--help'):
                text='''
Usage: python DrawingDIFF.py [-1 old_imagefile | -2 new_imagefile | -t | -o outputfile.tif | -s settingsfile ]
-1    : old input file name (also --old )
-2    : new input file name (also --new )
-t    : Create output TIF file and exit (must include old and/or new input filenames (also --tif )
-o    : output file name (also --out )
-s    : read settings file (also --settings )
-h    : print this help (also --help )
'''
                fmessage(text)
                sys.exit()
            #if option in ('-b','--batch'):
            #    self.batch = True

            if option in ('-s','--settings'):
                self.Open_Settings_File(value)
                
            if option in ('-1','--old'):
                self.Read_image_file_old(value)
                Load_Images=True

            if option in ('-2','--new'):
                self.Read_image_file_new(value)
                Load_Images=True

            if option in ('-o','--out'):
                temp_name, fileExtension = os.path.splitext(value)
                if fileExtension=='.tif' or fileExtension=='.tiff':
                    outputfile = value
                else:
                    outputfile = value+'.tif'     

            if option in ('-t','--tif'):
                self.batch = True
                make_TIF = True


                
        if Load_Images==True:
            self.menu_File_Compare_Images()
            if make_TIF:
                self.menu_File_Compare_Images()
                self.menu_File_Save(outputfile)
                sys.exit()
        ##########################################################################

        # Canvas
        #lbframe = Frame( self.master )
        #self.PreviewCanvas_frame = lbframe        
        #self.Zoom_Canvas = CanvasImage(lbframe, None)  # create widget

        self.PreviewCanvas_frame = Frame( self.master )        
        self.Zoom_Canvas = CanvasImage(self.PreviewCanvas_frame, None)  # create widget
        
        self.Zoom_Canvas.grid(row=0, column=0)
        self.Zoom_Canvas.canvas.configure(background="grey80")
        
        #self.Zoom_Canvas.canvas.bind("<1>"     , self.mouseLeftClick)
        #self.Zoom_Canvas.canvas.bind("<Motion>", self.mousePan)


        # make a Status Bar
        self.statusbar = Label(self.master, textvariable=self.statusMessage, bd=1, relief=SUNKEN , height=1)
        self.statusbar.pack(anchor=SW, fill=X, side=BOTTOM)
        

        # Buttons
        self.Save_Button = Button(self.master,text="Save",command=self.menu_File_Save)
        
        #self.Compare_Button      = Button(self.master,text="Update Display"     ,command=self.menu_File_Compare_Images)
        self.Open_New_Button     = Button(self.master,text="Open 'New' Image"   ,command=self.menu_File_Open_IMAGE_File_new)
        self.Open_Old_Button     = Button(self.master,text="Open 'Old' Image"   ,command=self.menu_File_Open_IMAGE_File_old)
        self.Clear_Image_Button  = Button(self.master,text="Clear Image Memory" ,command=self.menu_Image_File_Clear)
        self.Reload_Image_Button = Button(self.master,text="Reload Images"      ,command=self.Reload_images)
        self.Swap_Image_Button   = Button(self.master,text="Swap 'Old' & 'New'" ,command=self.Swap_images)


        # Left Column #
        self.Label_Options = Label(self.master,text="Options:", anchor=W)

        self.Label_highlight = Label(self.master,text="Highlight Changes")
        self.Checkbutton_highlight = Checkbutton(self.master,text=" ", anchor=W)
        self.Checkbutton_highlight.configure(variable=self.highlight)
        trace_variable(self.highlight, self.menu_File_Compare_Images)


        #self.Label_DRW_Options = Label(self.master,text="DRW Settings:", anchor=W)
        #self.Label_Sheet_Size      = Label(self.master,text="Sheet Size", anchor=CENTER )
        #self.Sheet_Size_OptionMenu = OptionMenu(root, self.sheet_size,
        #                                    "A",
        #                                    "B",
        #                                    "C",
        #                                    "D",
        #                                    "E",
        #                                    "F",command=self.Recalculate_RQD_Click)
        
        self.separator1 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator2 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator3 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        self.separator4 = Frame(self.master, height=2, bd=1, relief=SUNKEN)
        

        #-----------------------
        
        self.Label_Display_Ops = Label(self.master,text="Display Options:", anchor=W)
        self.Radio_Image_Old = Radiobutton(self.master,text="Show Old"  , value="old",width="100", anchor=W)
        self.Radio_Image_Old.configure(variable=self.image_display )

        self.Radio_Image_New = Radiobutton(self.master,text="Show New", value="new",width="100", anchor=W)
        self.Radio_Image_New.configure(variable=self.image_display )
        
        self.Radio_Image_Comb = Radiobutton(self.master,text="Show Both"  , value="comb", width="100", anchor=W)
        self.Radio_Image_Comb.configure(variable=self.image_display )
        
        trace_variable(self.image_display, self.menu_File_Compare_Images)

        self.Label_Old_File_Name = Label(self.master,text="Old File Name:", anchor=E)
        self.Label_New_File_Name = Label(self.master,text="New File Name:", anchor=E)
        self.Text_Old_File = Entry(self.master,textvariable=self.IMAGE_FILE_old)
        self.Text_New_File = Entry(self.master,textvariable=self.IMAGE_FILE_new)
        self.Text_Old_File.configure(bg=self.master.cget('bg'), relief=FLAT, state="readonly")
        self.Text_New_File.configure(bg=self.master.cget('bg'), relief=FLAT, state="readonly")

        # End Left Column #

        # Right Column     #
        # End Right Column #

        #GEN Setting Window Entry initializations
        self.Entry_Sspeed=Entry()
        self.Entry_BoxGap = Entry()
        self.Entry_ContAngle = Entry()
        self.Entry_Resolution = Entry()


        # Make Menu Bar
        self.menuBar = Menu(self.master, relief = "raised", bd=2)

        top_File = Menu(self.menuBar, tearoff=0)
        top_File.add("command", label = "Load Settings From File",  command = self.menu_File_Open_Settings_File)
        top_File.add("command", label = "Open Old Image File", command = self.menu_File_Open_IMAGE_File_old)
        top_File.add("command", label = "Open New Image File", command = self.menu_File_Open_IMAGE_File_new)
        #top_File.add("command", label = "Open New Image File", command = self.menu_File_Compare_Images)
        top_File.add("command", label = "Save",                command = self.menu_File_Save)

        top_File.add("command", label = "Exit", command = self.menu_File_Quit)
        self.menuBar.add("cascade", label="File", menu=top_File)

        self.top_Sheet = Menu(self.menuBar, tearoff=0)
        self.menuBar.add("cascade", label="Sheet", menu=self.top_Sheet)
        self.top_Sheet.add_separator()       
        trace_variable(self.sheet_number, self.View_Refresh_Callback)

        top_View = Menu(self.menuBar, tearoff=0)
        top_View.add("command", label = "Refresh", command = self.menu_View_Refresh)

        top_View.add_separator()
        top_View.add_checkbutton(label = "Highlight Differences", variable=self.highlight,  command= self.menu_View_Refresh)
        top_View.add_checkbutton(label = "Show Left Column", variable=self.show_col,  command= self.menu_View_Refresh)
        top_View.add_checkbutton(label = "Show File Names" , variable=self.show_row,  command= self.menu_View_Refresh)
        #top_View.add_checkbutton(label = "Show Grid"       , variable=self.show_grid, command= self.menu_View_Refresh)

        self.menuBar.add("cascade", label="View", menu=top_View)
        


        top_Settings = Menu(self.menuBar, tearoff=0)
        top_Settings.add("command", label = "General Settings", command = self.GEN_Settings_Window)

        self.menuBar.add("cascade", label="Settings", menu=top_Settings)

        top_Help = Menu(self.menuBar, tearoff=0)
        top_Help.add("command", label = "About", command = self.menu_Help_About)
        top_Help.add("command", label = "Help (Web Page)", command = self.menu_Help_Web)
        self.menuBar.add("cascade", label="Help", menu=top_Help)

        self.master.config(menu=self.menuBar)

        #was here
        ##########################################################################
        self.Zoom_Canvas.canvas.drop_target_register(tkinterdnd2.DND_FILES)
        self.Zoom_Canvas.canvas.dnd_bind('<<Drop>>', self.drop_new)
        
        self.Open_New_Button.drop_target_register(tkinterdnd2.DND_FILES)
        self.Open_New_Button.dnd_bind('<<Drop>>', self.drop_new)

        self.Open_Old_Button.drop_target_register(tkinterdnd2.DND_FILES)
        self.Open_Old_Button.dnd_bind('<<Drop>>', self.drop_old)

    # drop methods
    def drop(self,event):
        if event.data:
            files = self.Zoom_Canvas.canvas.tk.splitlist(event.data)
            if len(files)==1:
                self.Read_image_file_new(files[0])
                self.menu_File_Compare_Images()
            elif len(files)==2:
                self.Read_image_file_old(files[0])
                self.Read_image_file_new(files[1])
                self.menu_File_Compare_Images()
        return event.action
    
    def drop_new(self,event):
        if event.data:
            files = self.Zoom_Canvas.canvas.tk.splitlist(event.data)
            if len(files)==1:
                self.Read_image_file_new(files[0])
                self.menu_File_Compare_Images()
            elif len(files)==2:
                self.Read_image_file_old(files[0])
                self.Read_image_file_new(files[1])
                self.menu_File_Compare_Images()
        return event.action

    def drop_old(self,event):
        if event.data:
            files = self.Zoom_Canvas.canvas.tk.splitlist(event.data)
            if len(files)==1:
                self.Read_image_file_old(files[0])
                self.menu_File_Compare_Images()
            elif len(files)==2:
                self.Read_image_file_old(files[0])
                self.Read_image_file_new(files[1])
                self.menu_File_Compare_Images()
        return event.action



    def status_message(self,message_text,color='white'):
        if not self.batch:
            self.statusbar.configure( bg = color )
            self.statusMessage.set(message_text)
            self.update_idletasks()
        
################################################################################
    def entry_set(self, val2, calc_flag=0, new=0):
        if calc_flag == 0 and new==0:
            try:
                self.statusbar.configure( bg = 'yellow' )
                val2.configure( bg = 'yellow' )
                self.statusMessage.set(" Recalculation required.")
            except:
                pass
        elif calc_flag == 3:
            try:
                val2.configure( bg = 'red' )
                self.statusbar.configure( bg = 'red' )
                self.statusMessage.set(" Value should be a number. ")
            except:
                pass
        elif calc_flag == 2:
            try:
                self.statusbar.configure( bg = 'red' )
                val2.configure( bg = 'red' )
            except:
                pass
        elif (calc_flag == 0 or calc_flag == 1) and new==1 :
            try:
                self.statusbar.configure( bg = 'white' )
                self.statusMessage.set(" ")
                val2.configure( bg = 'white' )
            except:
                pass
        elif (calc_flag == 1) and new==0 :
            try:
                self.statusbar.configure( bg = 'white' )
                self.statusMessage.set(" ")
                val2.configure( bg = 'white' )
            except:
                pass

        elif (calc_flag == 0 or calc_flag == 1) and new==2:
            return 0
        return 1

################################################################################
    def Write_Config_File(self, event):
        self.Write_Settings(rough_flag=0,config_file=True)
        config_file = "DrawingDIFF.txt"
        configname_full = self.HOME_DIR + "/" + config_file

        win_id=self.grab_current()
        if ( os.path.isfile(configname_full) ):
            if not message_ask_ok_cancel("Replace", "Replace Exiting Configuration File?\n"+configname_full):
                try:
                    win_id.withdraw()
                    win_id.deiconify()
                except:
                    pass
                return

        try:
            fout = open(configname_full,'w')
        except:
            self.statusMessage.set("Unable to open file for writing: %s" %(configname_full))
            self.statusbar.configure( bg = 'red' )
            return
        for line in self.settings:
            try:
                fout.write(line+'\n')
            except:
                fout.write('(skipping line)\n')
        fout.close
        self.statusMessage.set("Configuration File Saved: %s" %(configname_full))
        self.statusbar.configure( bg = 'white' )


    ################################################################################
    def Write_Settings(self,rough_flag = 0,config_file=False):
        global Zero
        header = []
        header.append('( DrawingDIFF Settings: '+version+' )')
        header.append('( by Scorch - 2021-2026 )')
        header.append("(=========================================================)")
        # BOOL
        header.append('(DrawingDIFF_set highlight  %s )' %( int(self.highlight.get()) ))
        #header.append('(DrawingDIFF_set show_grid  %s )' %( int(self.show_grid.get()) ))
        header.append('(DrawingDIFF_set show_col   %s )' %( int(self.show_col.get())  ))
        header.append('(DrawingDIFF_set show_row   %s )' %( int(self.show_row.get())  ))

        # STRING
        #header.append('(DrawingDIFF_set Sheet_Size  %s )'  %( self.sheet_size.get()          ))        
        header.append('(DrawingDIFF_set resolution  %s )'  %( self.resolution.get()          ))
        header.append('(DrawingDIFF_set image_display  %s )'  %( self.image_display.get()    ))

        # QUOTED STRING
        header.append('(DrawingDIFF_set gs_path         \042%s\042 )' %( self.gs_path.get()  ))       
        header.append('(DrawingDIFF_set IMAGE_FILE_old  \042%s\042 )' %( self.IMAGE_FILE_old.get() ))
        header.append('(DrawingDIFF_set IMAGE_FILE_new  \042%s\042 )' %( self.IMAGE_FILE_new.get() ))

        # COLORS
        header.append('(DrawingDIFF_set OLD_COLOR        %03d,%03d,%03d,%03d )'  %( self.OLD_COLOR       ))
        header.append('(DrawingDIFF_set NEW_COLOR        %03d,%03d,%03d,%03d )'  %( self.NEW_COLOR       ))
        header.append('(DrawingDIFF_set BOTH_COLOR       %03d,%03d,%03d,%03d )'  %( self.BOTH_COLOR      ))
        header.append('(DrawingDIFF_set HIGHLIGHT_COLOR  %03d,%03d,%03d,%03d )'  %( self.HIGHLIGHT_COLOR ))
        
        header.append("(=========================================================)")

        if (config_file==True):
            self.settings = []
            self.settings = header
            return
        ######################################################

    def Quit_Click(self, event):
        self.statusMessage.set("Exiting!")
        root.destroy()

    def make_handle(self,x,y,thick=1,radius=10,handle_tags="handle",col="blue"):
        self.Zoom_Canvas.canvas.create_oval( x-radius, y-radius, x+radius, y+radius,
                                             fill=col, outline="black", width = thick, tags=handle_tags)

    def plot_polygon(self,coords):
        self.Zoom_Canvas.canvas.create_polygon( coords,fill="blue",stipple='gray12',outline="black",width = 1,tags="poly")


    def coords2inches(self,x,y):
        x_img  = float(self.Zoom_Canvas.image_location[0])
        y_img  = float(self.Zoom_Canvas.image_location[1])
        x_img1 = float(self.Zoom_Canvas.image_location[2])
        y_img1 = float(self.Zoom_Canvas.image_location[3])
        scale  = float(self.Zoom_Canvas.image_location[4])
        x_out = ((float(x)-x_img+x_img1)/scale)/float(self.resolution.get())
        y_out = ((float(y)-y_img+y_img1)/scale)/float(self.resolution.get())
        return [x_out,y_out]

    def plot_polygon_new(self,coords):
        x_img  = float(self.Zoom_Canvas.image_location[0])
        y_img  = float(self.Zoom_Canvas.image_location[1])
        x_img1 = float(self.Zoom_Canvas.image_location[2])
        y_img1 = float(self.Zoom_Canvas.image_location[3])
        scale  = float(self.Zoom_Canvas.image_location[4])
        can_coords=[]
        scale_val = float(self.resolution.get())*scale
        x_shift=+x_img-x_img1
        y_shift=+y_img-y_img1
        for point in coords:
            xpt = point[0]*scale_val+x_shift
            ypt = point[1]*scale_val+y_shift
            can_coords.append([xpt,ypt])

        self.plot_polygon(can_coords)
        
    def mouseLeftClick(self,event):
        x = self.Zoom_Canvas.canvas.canvasx(event.x)
        y = self.Zoom_Canvas.canvas.canvasy(event.y)

        #print(self.Zoom_Canvas.canvas.yview)
        if self.making_shape==False: # Start a new shape
            self.shape_coords=[]
            self.first_x = x
            self.first_y = y
            self.start_x = x
            self.start_y = y


            #[x_out,y_out] = self.coords2inches(x,y)
            #print(x_out,y_out)
            #self.shape_coords.append([x,y])            
            self.shape_coords.append(self.coords2inches(x,y))
            self.make_handle(x,y)
            self.make_handle(x,y,col="red",handle_tags=("handle","1st_handle"))
            self.making_shape=True
            
        else: # Operate on an existing shape
            current_tags =self.Zoom_Canvas.canvas.gettags(CURRENT)
            if "1st_handle" in current_tags: # Finish the current shape
                self.Zoom_Canvas.canvas.create_line( self.start_x,self.start_y,self.end_x,self.end_y,fill="red",width = 10,tags="line")
                self.making_shape=False

                self.Zoom_Canvas.canvas.delete("new_line")
                self.Zoom_Canvas.canvas.delete("line")
                self.Zoom_Canvas.canvas.delete("handle")
                self.plot_polygon_new(self.shape_coords)
                #self.Zoom_Canvas.canvas.create_polygon( self.shape_coords,fill="blue",stipple='gray12',outline="black",width = 1,tags="poly")
                self.Notation_boxes.append(Notation_Box(self.shape_coords))

                #first_handles = self.Zoom_Canvas.canvas.find_withtag("1st_handle")
                #for canvas_id in first_handles:
                #    self.Zoom_Canvas.canvas.delete(canvas_id)
                
            else: # Continue an existing shape
                #self.shape_coords.append([x,y])
                self.shape_coords.append(self.coords2inches(x,y))
                self.Zoom_Canvas.canvas.create_line( self.start_x,self.start_y,x,y,fill="red",width = 10,tags="line")
                self.start_x = x
                self.start_y = y
                self.make_handle(x,y)
        self.Zoom_Canvas.canvas.tag_raise("handle")
        self.Zoom_Canvas.canvas.tag_raise("1st_handle")
            
        
    def mousePan(self,event):
        if self.making_shape==True:
            self.end_x = self.Zoom_Canvas.canvas.canvasx(event.x)
            self.end_y = self.Zoom_Canvas.canvas.canvasy(event.y)
            self.Zoom_Canvas.canvas.delete("new_line")
            self.Zoom_Canvas.canvas.create_line( self.start_x,self.start_y,self.end_x,self.end_y,fill="red",width = 10,tags="new_line")
            self.Zoom_Canvas.canvas.tag_raise("handle")

    def Recalculate_Click(self, event):
        pass

    def Settings_ReLoad_Click(self, event):
        win_id=self.grab_current()

    def Close_Current_Window_Click(self):
        win_id=self.grab_current()
        win_id.destroy()

    def Stop_Click(self, event):
        global STOP_CALC
        STOP_CALC=1
        
    # Left Column #
    #############################
       

    #############################
    # End Left Column #
    #############################
    
    #############################
    def Entry_Resolution_Check(self):
        try:
            value = float(self.resolution.get())
            if  value <= 0.0:
                self.statusMessage.set(" Resolution should be greater than 0 ")
                return 2 # Value is invalid number
        except:
            return 3     # Value not a number
        return 0         # Value is a valid number
    def Entry_Resolution_Callback(self, varName, index, mode):
        self.entry_set(self.Entry_Resolution,self.Entry_Resolution_Check(), new=1)
    #############################

    ##########################################################################
    ##########################################################################
    def Check_All_Variables(self):
        MAIN_error_cnt= 0

        GEN_error_cnt= \
        self.entry_set(self.Entry_Resolution,self.Entry_Resolution_Check(),2)

        ERROR_cnt = MAIN_error_cnt + GEN_error_cnt

        if (ERROR_cnt > 0):
            self.statusbar.configure( bg = 'red' )
        if (GEN_error_cnt > 0):
            self.statusMessage.set(\
                " Entry Error Detected: Check Entry Values in General Settings Window ")
        if (MAIN_error_cnt > 0):
            self.statusMessage.set(\
                " Entry Error Detected: Check Entry Values in Main Window ")

        return ERROR_cnt


    def menu_File_Open_Settings_File(self):
        #if ( not os.path.isdir(init_dir) ):
        init_dir = os.path.expanduser("~")
        fileselect = askopenfilename(filetypes=[("Settings Files","*.txt"),\
                                                ("All Files","*")],\
                                                 initialdir=init_dir)
        if fileselect != '' and fileselect != ():
            self.Open_Settings_File(fileselect)


##    def Make_Images_From_PDF_multi_sheet(self, input_path=None):
##        img = None
##        
##        dpi = int(self.resolution.get())
##        res    = "-r%dx%d" %(dpi,dpi)
##        #create OS temp folder
##        tmp_dir   = tempfile.mkdtemp()
##        temp_file = os.path.join(tmp_dir, "out.tif")
##        try:
##            args = [
##                "not_used", # actual value doesn't matter
##                "-q",
##                res,
##                "-dNOPAUSE",
##                "-dBATCH",
##                #"-dSAFER",
##                "-sDEVICE=tiffpack",
##                "-sOutputFile=%s" %(temp_file),
##                "-f",  input_path
##                ]
##            # arguments have to be bytes, encode them
##            encoding = locale.getpreferredencoding()
##            args = [a.encode(encoding) for a in args]
##            GS = ghostscript.Ghostscript(*args)
##            GS.exit()
##        except Exception as e:
##            try:
##                GS.exit()
##            except:
##                pass
##            self.remove_dir(tmp_dir)
##            error_text = "%s" %(e)
##            raise Exception("Ghostscript Execution Failed.\n%s" %(error_text))
##
##        try:
##            im = Image.open(temp_file)
##            PIL_im = []
##            for i, page in enumerate(ImageSequence.Iterator(im)):
##                PIL_im.append(page.convert("L"))
##            im.close()
##        except:
##            self.remove_dir(tmp_dir)
##            raise Exception("Failed to read image data:\n%s" %(error_text))
##            
##        self.remove_dir(tmp_dir)
##        return PIL_im

    def Make_Images_From_PDF_multi_sheet(self, gs_exe_path=None, input_path=None):
        img = None
        #create OS temp folder
        tmp_dir = tempfile.mkdtemp()
        temp_file = os.path.join(tmp_dir, "out.tif")
        if os.path.exists(gs_exe_path):
            try:
                dpi = int(self.resolution.get())
                res    = "-r%dx%d" %(dpi,dpi)
                out_path = "-sOutputFile=%s" %(temp_file)
                cmd = [ gs_exe_path,"-q",\
                                    res,\
                                    "-dNOPAUSE",\
                                    "-sDEVICE=tiffpack",\
                                    "-dBATCH",\
                                    out_path,\
                                    input_path]
                self.timout = 60
                stdout,stderr=run_external(cmd, self.timout)
                #print("stdout\n-----\n",stdout)
                #print("stderr\n-----\n",stderr)
            except Exception as e:
                self.remove_dir(tmp_dir)
                error_text = "%s" %(e)
                raise Exception("Ghostscript Execution Failed.\n%s" %(error_text))
        else:
            self.remove_dir(tmp_dir) 
            raise Exception("Ghostscript Not found.")
        im = Image.open(temp_file)
        PIL_im = []
        for i, page in enumerate(ImageSequence.Iterator(im)):
            PIL_im.append(page.convert("L"))
        im.close()
        
        self.remove_dir(tmp_dir)
        return PIL_im
    
    def remove_dir(self,tmp_dir):
        try:
            shutil.rmtree(tmp_dir) 
        except:
            raise Exception("Temp dir failed to delete:\n%s" %(tmp_dir) )
            debug_message(traceback.format_exc())
    
    def menu_File_Open_IMAGE_File_old(self):        
        init_dir = os.path.dirname(self.IMAGE_FILE_old.get())
        if ( not os.path.isdir(init_dir) ):
            init_dir = os.path.expanduser("~")
        fileselect = askopenfilename(filetypes=[("PDF Files", ("*.pdf")),\
                                                ("Image Files", ("*.png","*.tif")),\
                                                ("All Files","*")],\
                                                initialdir=init_dir)
        if fileselect != '' and fileselect != ():
            self.Read_image_file_old(fileselect)
            self.menu_File_Compare_Images()


    def menu_File_Open_IMAGE_File_new(self):
        init_dir = os.path.dirname(self.IMAGE_FILE_new.get())
        if ( not os.path.isdir(init_dir) ):
            init_dir = os.path.expanduser("~")
        fileselect = askopenfilename(filetypes=[("PDF Files", ("*.pdf")),\
                                                ("Image Files", ("*.png","*.tif")),\
                                                ("All Files","*")],\
                                                initialdir=init_dir)
        if fileselect != '' and fileselect != ():
            self.Read_image_file_new(fileselect)
            self.menu_File_Compare_Images()

    def menu_File_Compare_Images(self,a=None,b=None,c=None):
        im_display = self.image_display.get()
        im_comp = []
        
        
        n_sheets = max(len(self.im_old),len(self.im_new))
        for i in range(n_sheets):
            self.status_message("Processing Sheet %d of %d" %(i+1,n_sheets),'yellow')
            try:
                old = self.im_old[i]
            except:
                old = []
            try:
                new = self.im_new[i]
            except:
                new = []
            
    
            if im_display=="old":
                OLD_COLOR  = self.OLD_COLOR
                NEW_COLOR  = (255,255,255,255)
                BOTH_COLOR = self.BOTH_COLOR
                
            if im_display=="new":
                OLD_COLOR  = (255,255,255,255)
                NEW_COLOR  = self.NEW_COLOR
                BOTH_COLOR = self.BOTH_COLOR
                
            if im_display=="comb":
                OLD_COLOR  = self.OLD_COLOR
                NEW_COLOR  = self.NEW_COLOR
                BOTH_COLOR = self.BOTH_COLOR
                
            #if self.highlight.get():
            #BOTH_COLOR = (0  ,0  ,0  ,255)    


    
            start = time.time()
            im_comp.append( image_op_blur(old,
                                          new,
                                          HIGHLIGHT=self.highlight.get(),
                                          OLD_COLOR=OLD_COLOR,
                                          NEW_COLOR=NEW_COLOR,
                                          BOTH_COLOR=BOTH_COLOR,
                                          HIGHLIGHT_COLOR=self.HIGHLIGHT_COLOR,
                                          n=10))
            #print("Elapsed Time = %.2f" %(time.time()-start ))
        if n_sheets>1:
            self.status_message("Done Processing %d Sheets" %(n_sheets))
        else:
            self.status_message("Done Processing %d Sheet" %(n_sheets))
        self.im = im_comp
        if not self.batch:
            self.reset_sheet_numbers()
            self.Plot_Data()
        

    def menu_Image_File_Clear(self):
        self.sheet_number.set("0")
        self.im_old=[]
        self.im_new=[]
        self.im    =[]
        self.reset_sheet_numbers()
        self.menu_File_Compare_Images()
        

    def reset_sheet_numbers(self):
        ## Remove Sheet Numbers
        for i in range(1000):
            self.top_Sheet.delete(0)
            if "" == self.top_Sheet.entrycget(0, "label"):
                break
        ## Add Sheet Numbers
        for i in range(len(self.im)):
            val="%d"%(i)
            lab="Sheet %d"%(i+1)
            self.top_Sheet.add_radiobutton(label = lab ,   variable=self.sheet_number, value=val)
        self.sheet_number.set("0")

    def color_from_csv(self,csv):
        color_str = csv.split(',')
        COLOR = (int(color_str[0]),int(color_str[1]),int(color_str[2]),int(color_str[3]))
        return COLOR

    def Open_Settings_File(self,filename):
        try:
            fin = open(filename,'r')
        except:
            fmessage("Unable to open file: %s" %(filename))
            return
        
        text_codes=[]
        ident = "DrawingDIFF_set"
        for line in fin:
            if ident in line:
                # BOOL
                if "highlight"  in line:
                    self.highlight.set(line[line.find("highlight"):].split()[1])
                if "show_grid"  in line:
                    self.show_grid.set(line[line.find("show_grid"):].split()[1])
                if "show_row"  in line:
                    self.show_row.set(line[line.find("show_row"):].split()[1])
                if "show_col"  in line:
                    self.show_col.set(line[line.find("show_col"):].split()[1])
                    
                # STRING
                elif "Sheet_Size"    in line:
                    self.sheet_size.set(line[line.find("Sheet_Size"):].split()[1])   
                elif "resolution"    in line:
                     self.resolution.set(line[line.find("resolution"):].split()[1])
                elif "image_display"    in line:
                     self.image_display.set(line[line.find("image_display"):].split()[1])

                # QUOTED STRING
                elif "gs_path"    in line:
                     self.gs_path.set(line[line.find("gs_path"):].split("\042")[1])
                elif "IMAGE_FILE_old"    in line:
                       self.IMAGE_FILE_old.set(line[line.find("IMAGE_FILE_old"):].split("\042")[1])
                elif "IMAGE_FILE_new"    in line:
                       self.IMAGE_FILE_new.set(line[line.find("IMAGE_FILE_new"):].split("\042")[1])

                # COLORS
                elif "OLD_COLOR"    in line:
                    self.OLD_COLOR = self.color_from_csv( line[line.find("OLD_COLOR"):].split()[1] )
                elif "NEW_COLOR"    in line:
                    self.NEW_COLOR = self.color_from_csv( line[line.find("NEW_COLOR"):].split()[1] )
                elif "BOTH_COLOR"    in line:
                    self.BOTH_COLOR = self.color_from_csv( line[line.find("BOTH_COLOR"):].split()[1] )
                elif "HIGHLIGHT_COLOR"    in line:
                    self.HIGHLIGHT_COLOR = self.color_from_csv( line[line.find("HIGHLIGHT_COLOR"):].split()[1] )
                       
        fin.close()
        
        temp_name, fileExtension = os.path.splitext(filename)
        file_base=os.path.basename(temp_name)
            
        if self.initComplete == 1:
            self.menu_Mode_Change()
            self.Reload_images()
        
    ###############################################


    def Reload_images(self):
        fileName, fileExtension = os.path.splitext(self.IMAGE_FILE_old.get())
        init_file=os.path.basename(fileName)
        if init_file != "None":
            if ( os.path.isfile(self.IMAGE_FILE_old.get()) ):
                self.Read_image_file_old(self.IMAGE_FILE_old.get())
            else:
                self.statusMessage.set("Image file not found: %s " %(self.IMAGE_FILE_old.get()))

        fileName, fileExtension = os.path.splitext(self.IMAGE_FILE_new.get())
        init_file=os.path.basename(fileName)
        if init_file != "None":
            if ( os.path.isfile(self.IMAGE_FILE_new.get()) ):
                self.Read_image_file_new(self.IMAGE_FILE_new.get())
            else:
                self.statusMessage.set("Image file not found: %s " %(self.IMAGE_FILE_new.get()))
        self.menu_File_Compare_Images()


    def Swap_images(self):
        # store previous old image data
        PIL_im_temp_old = self.im_old
        filename_temp_old = self.IMAGE_FILE_old.get()
        
        # set old image
        PIL_im = self.im_new
        if len(PIL_im)>0:
            self.wim_old, self.him_old = PIL_im[0].size
            self.aspect_ratio_old =  float(self.wim_old-1) / float(self.him_old-1)
            self.im_old = PIL_im
            self.IMAGE_FILE_old.set( self.IMAGE_FILE_new.get() )
        else:
            self.IMAGE_FILE_old.set(os.path.expanduser("~")+"/None")

        
        #Set new image
        PIL_im = PIL_im_temp_old
        if len(PIL_im)>0:
            self.wim_new, self.him_new = PIL_im[0].size
            self.aspect_ratio_new =  float(self.wim_new-1) / float(self.him_new-1)
            self.im_new = PIL_im
            self.IMAGE_FILE_new.set(filename_temp_old)
        else:
            self.IMAGE_FILE_new.set(os.path.expanduser("~")+"/None")
            
        self.menu_File_Compare_Images()

    def Read_image_file_old(self,fileselect):
        PIL_im = self.Read_image_file(fileselect)
        if len(PIL_im)>0:
            self.wim_old, self.him_old = PIL_im[0].size
            self.aspect_ratio_old =  float(self.wim_old-1) / float(self.him_old-1)
            self.im_old = PIL_im
            self.IMAGE_FILE_old.set(fileselect)


    def Read_image_file_new(self,fileselect):
        PIL_im = self.Read_image_file(fileselect)
        if len(PIL_im)>0:
            self.wim_new, self.him_new = PIL_im[0].size
            self.aspect_ratio_new =  float(self.wim_new-1) / float(self.him_new-1)
            self.im_new = PIL_im
            self.IMAGE_FILE_new.set(fileselect)

    def Read_image_file(self,fileselect):
        PIL_im = []
        if not ( os.path.isfile(fileselect) ):
            self.status_message("Image file not found: %s" %(fileselect),'red')
            fmessage("Image file not found: %s" %(fileselect))
        else:
            self.status_message("Loading Image File: %s" %(fileselect))
            try:
                fileName, fileExtension = os.path.splitext(fileselect)        
                TYPE=fileExtension.upper()
                if TYPE=='.PDF':
                    PIL_im = self.Make_Images_From_PDF_multi_sheet(gs_exe_path=self.gs_path.get(), input_path=fileselect)   
                #if TYPE=='.PDF':
                #    PIL_im = self.Make_Images_From_PDF_multi_sheet(input_path=fileselect)
                else:
                    im = Image.open(fileselect)
                    for i, page in enumerate(ImageSequence.Iterator(im)):
                        PIL_im.append(page.convert("L"))
            except Exception as e:
                msg1 = "Image read failed: "
                msg2 = "%s" %(e)
                self.status_message((msg1+msg2).split("\n")[0],color='red')
                fmessage((msg1+msg2).split("\n")[0])
                debug_message(traceback.format_exc())
        return PIL_im
                
            
    ##########################################################################
    ##########################################################################
    #def menu_File_Save(self):
    #    self.WriteSVG()
    #    fmessage("File Saved")
    
    #def menu_File_Save(self):
    #    #self.WriteSVG()
    #    self.Write_Image()
    #    fmessage("File Saved")


    ##########################################################################
    def menu_File_Save(self, filename=None):
        OldName = self.IMAGE_FILE_old.get()
        NewName = self.IMAGE_FILE_new.get()
        if len(self.im)==0:
            self.status_message("No image data to save.", 'white')
            return

        init_dir = os.path.dirname(NewName)
        if ( not os.path.isdir(init_dir) ):
            init_dir = self.HOME_DIR
            
        OldfileName, OldfileExtension = os.path.splitext(OldName)
        NewfileName, NewfileExtension = os.path.splitext(NewName)
        old_init_file=os.path.basename(OldfileName)
        new_init_file=os.path.basename(NewfileName)
        init_file = old_init_file+"_vs_"+new_init_file

        if filename == "default":
            filename = init_file+".tif"

        if (filename==None):
            filename = asksaveasfilename(defaultextension='.tif', \
                                         filetypes=[("TIFF Image","*.tif")],\
                                         initialdir=init_dir,\
                                         initialfile= init_file )
    
        if filename != '' and filename != ():            
            self.Write_Image(filename)
            self.status_message("File Saved: %s" %(filename), 'white')
    ##########################################################################
    ##########################################################################
            
    def menu_File_Quit(self):
        if message_ask_ok_cancel("Exit", "Exiting...."):
            self.Quit_Click(None)

    def menu_View_Refresh_Callback(self, varName, index, mode):
        self.menu_View_Refresh()

    def menu_View_Refresh(self):
        dummy_event = Event()
        dummy_event.widget=self.master
        self.Master_Configure(dummy_event,1)
        self.Plot_Data()

    def menu_Mode_Change_Callback(self, varName, index, mode):
        self.menu_View_Refresh()

    def menu_Mode_Change(self):
        dummy_event = Event()
        dummy_event.widget=self.master
        self.Master_Configure(dummy_event,1)

    def menu_View_Recalculate(self):
        pass

    def menu_Help_About(self):
        application="DrawingDIFF"
        about = "%s Version %s\n\n" %(application,version)
        about = about + "By Scorch.\n"
        about = about + "\163\143\157\162\143\150\100\163\143\157\162"
        about = about + "\143\150\167\157\162\153\163\056\143\157\155\n"
        about = about + "https://www.scorchworks.com/\n\n"
        try:
            python_version = "%d.%d.%d" %(sys.version_info.major,sys.version_info.minor,sys.version_info.micro)
        except:
            python_version = ""
        about = about + "Python "+python_version+" (%d bit)" %(struct.calcsize("P") * 8)
        message_box("About %s" %(application),about)

    def menu_Help_Web(self):
        fmessage("No webpage yet.")
        webbrowser.open_new(r"http://www.scorchworks.com/drawingdiff/drawingdiff.html.html")

    def KEY_ESC(self, event):
        pass #A stop calculation command may go here

    def KEY_F1(self, event):
        self.menu_Help_About()

    def KEY_F2(self, event):
        self.GEN_Settings_Window()

    def KEY_F3(self, event):
        fmessage("F3 Pressed")

    def KEY_F4(self, event):
        pass
        

    def KEY_F5(self, event):
        #self.Zoom_Canvas.home_image()
        self.menu_View_Refresh()

    def page_up(self,event=None):
        current_sheet = int(self.sheet_number.get())
        if current_sheet < len(self.im)-1:
            self.sheet_number.set("%d" %(current_sheet+1))
            self.menu_View_Refresh()
        
    def page_down(self,event=None):
        current_sheet = int(self.sheet_number.get())
        if current_sheet > 0:
            self.sheet_number.set("%d" %(current_sheet-1))
            self.menu_View_Refresh()
        
    def bindConfigure(self, event):
        if not self.initComplete:
            self.initComplete = 1
            self.menu_Mode_Change()
            self.Reload_images()

    def Master_Configure(self, event, update=0):
        if event.widget != self.master or self.initComplete != 1:
            return
        x = int(self.master.winfo_x())
        y = int(self.master.winfo_y())
        w = int(self.master.winfo_width())
        h = int(self.master.winfo_height())
        if (self.x, self.y) == (-1,-1):
            self.x, self.y = x,y
        if abs(self.w-w)>10 or abs(self.h-h)>10 or update==1:
            ###################################################
            #  Form changed Size (resized) adjust as required #
            ###################################################
            self.w=w
            self.h=h

            #####################################
            #          Left Column              #
            #####################################
            if self.show_col.get():
                left_col  = 185 # Set to 20 for no column
                wide_item = 160
                w_label=90
                w_entry=60-35
                w_units=35

                x_label_L=10
                x_entry_L=x_label_L+w_label+10
                x_units_L=x_entry_L+w_entry+5

                Yloc=6
                #self.Label_DRW_Options.place(x=x_label_L, y=Yloc, width=wide_item, height=21)
                #Yloc=Yloc+24
                #self.Label_Sheet_Size.place(x=x_label_L, y=Yloc, width=w_label, height=21)
                #self.Sheet_Size_OptionMenu.place(x=x_entry_L, y=Yloc, width=60, height=23)
                #Yloc=Yloc+24+12
                #self.separator1.place(x=x_label_L, y=Yloc,width=wide_item, height=2)
                #Yloc=Yloc+6
                #self.Label_Options.place(x=x_label_L, y=Yloc, width=wide_item, height=21)
                #Yloc=Yloc+24

                self.Open_Old_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                self.Open_New_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                self.Label_Display_Ops.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24
                

                self.Radio_Image_Old .place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24
                
                self.Radio_Image_New.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24
                
                self.Radio_Image_Comb.place(x=12, y=Yloc, width=wide_item, height=30) 
                Yloc=Yloc+24+12

                self.Label_highlight.place(x=x_label_L, y=Yloc, width=w_label+20, height=21)
                self.Checkbutton_highlight.place(x=x_entry_L+20, y=Yloc, width=w_entry+20, height=23)

                Yloc=Yloc+24+12
                self.separator2.place(x=x_label_L, y=Yloc,width=wide_item, height=2)
                Yloc=Yloc+12
                
                
                #self.Compare_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                #Yloc=Yloc+24+12
                
                
                self.Swap_Image_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                self.Reload_Image_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

                self.Clear_Image_Button.place(x=12, y=Yloc, width=wide_item, height=30)
                Yloc=Yloc+24+12

        
                
                
                #Ybut=self.h-60
                # Button
                #self.Save_Button.place(x=12, y=Ybut, width=95, height=30)
        
            else:
                left_col = 20
                #self.Label_DRW_Options.place_forget()
                #self.Label_Sheet_Size.place_forget()
                #self.Sheet_Size_OptionMenu.place_forget()
                #self.separator1.place_forget()
                #self.Label_Options.place_forget()
                self.Label_highlight.place_forget()
                self.Checkbutton_highlight.place_forget()
                self.separator2.place_forget()
                #self.Compare_Button.place_forget()
                self.Open_New_Button.place_forget()
                self.Open_Old_Button.place_forget()
                self.Clear_Image_Button.place_forget()
                self.Reload_Image_Button.place_forget()
                self.Swap_Image_Button.place_forget()
                self.Label_Display_Ops.place_forget()
                self.Radio_Image_Old.place_forget()
                self.Radio_Image_New.place_forget()
                self.Radio_Image_Comb.place_forget()
                self.Save_Button.place_forget()
            #####################################
            #       End Left Column             #
            #####################################

            #####################################
            #          Bottom Row               #
            #####################################
            if self.show_row.get():
                bot_row = 100
                Yloc=self.h-50
                text_pos=100
                self.Label_Old_File_Name.place(x=0, y=Yloc, width=text_pos, height=30)
                self.Text_Old_File.place(x=text_pos+10, y=Yloc+5, width=self.w-20-text_pos)
                
                Yloc=Yloc-20
                self.Label_New_File_Name.place(x=0, y=Yloc, width=text_pos, height=30)
                self.Text_New_File.place(x=text_pos+10, y=Yloc+5, width=self.w-20-text_pos)
                
            else:
                bot_row  = 50  # Set to 50 for no row
                self.Label_Old_File_Name.place_forget()
                self.Text_Old_File.place_forget()
                self.Label_New_File_Name.place_forget()
                self.Text_New_File.place_forget()
            #####################################
            #        End Bottom Row             #
            #####################################

            canWidth  = self.w-(left_col+20)
            canHeight = self.h-bot_row

            try:
                self.Zoom_Canvas.canvas.configure( width = canWidth, height = canHeight ) #NEW
            except:
                print("can't set size")
                pass
            self.Zoom_Canvas.home_image()
            
            self.PreviewCanvas_frame.place(x=left_col, y=10)  
            self.Set_Input_States()

    def View_Refresh_Callback(self, varName, index, mode):
        self.menu_View_Refresh()
        
    def Recalculate_RQD_Click(self, event):
        self.menu_View_Refresh()

    def Set_Input_States(self):
        pass
            
    def Set_Input_States_Event(self,event):
        self.Set_Input_States()
            
    def Set_Input_States_GEN_Event(self,event):
        self.Set_Input_States_GEN()

    def Write_Image(self,filename):
        temp_name, fileExtension = os.path.splitext(filename)

        if fileExtension=='.tif' or fileExtension=='.tiff':
            self.im[0].save(filename, compression="tiff_deflate", save_all=True, append_images=self.im[1:])
        else:
            for i in range(len(self.im)):
                self.im[i].save(temp_name+"_%03d"%(i)+fileExtension)
        

    def WriteSVG(self):
        Thick = 0.01
        dpi=float(self.resolution.get())
        width,height  = self.im[0].size
        width_in  = float(width)  / float(dpi)
        height_in = float(height) / float(dpi)
        
        self.svgcode = []
        
        ## Create SVG Header Data
        self.svgcode.append('<?xml version="1.0" standalone="no"?>')
        self.svgcode.append('<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN"  ')
        self.svgcode.append('  "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">  ')
        self.svgcode.append('<svg width="%f%s" height="%f%s" viewBox="0 0 %f %f"  ' \
                            %(width_in,"in",height_in,"in",width,height) )
        self.svgcode.append('     xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" version="1.1">')
        self.svgcode.append('  <title> F-engrave Output </title>')
        self.svgcode.append('  <desc>SVG File Created By DrawingDIFF</desc>')


        # Add Image
        self.svgcode.append('<image')
        self.svgcode.append('y="1"')
        self.svgcode.append('x="1"')
        self.svgcode.append('xlink:href="data:image/png;base64,')

        b = io.BytesIO()
        self.im[0].save(b,'PNG')
        image_bytes = b.getvalue()
        str = base64.b64encode(image_bytes)
        
        self.svgcode.append(str)
        self.svgcode.append('"')
        self.svgcode.append('preserveAspectRatio="none"')
        self.svgcode.append('height="%f"' %(height))
        self.svgcode.append('width="%f" />' %(width))


        ## Make SVG Polygons
        for i in range(len(self.Notation_boxes)):             
            self.svgcode.append('  <polygon points="')
            for j in range( len(self.Notation_boxes[i].coords)):
                self.svgcode.append('%f %f ' %(self.Notation_boxes[i].coords[j][0]*dpi,self.Notation_boxes[i].coords[j][1]*dpi))
            self.svgcode.append('"        fill="blue" fill-opacity="0.16" stroke="blue" stroke-width="%f" stroke-linecap="round" stroke-linejoin="round">' %(Thick*dpi))

            # Add Description
            self.svgcode.append('<desc>%s</desc>' %('new custom \n description'))
            # Add Title
            self.svgcode.append('<title>%s</title>' %("new custom title"))

            
            self.svgcode.append('</polygon>')
        ## End SVG Polygons
            
        self.svgcode.append('</svg>')

        ## Write SVG data to file
        #fout = open("Z:\out.svg",'w')
        for line in self.svgcode:
            fout.write(line+'\n')
            
        fout.close()

    ##########################################
    #        CANVAS PLOTTING STUFF           #
    ##########################################
    def Plot_Data(self):
        self.Zoom_Canvas.canvas.delete(ALL)
        #self.Zoom_Canvas.canvas.create_line( 0,0,400,400,fill="red",width = 10)
            
        if (self.Check_All_Variables() > 0):
            return
        
        cszw = int(self.Zoom_Canvas.canvas.cget("width"))
        cszh = int(self.Zoom_Canvas.canvas.cget("height"))

        wc = float(cszw/2)
        hc = float(cszh/2)
        
        sheet = int(self.sheet_number.get())
##        if self.im!=[]:
##            try:
##                test = self.im[sheet].size
##                self.SCALE = min( float(cszw-20)/float(self.wim), float(cszh-20)/float(self.him))
##                if self.SCALE < 1:
##                    nw=int(self.SCALE*self.wim)
##                    nh=int(self.SCALE*self.him)
##                else:
##                    nw = self.wim
##                    nh = self.him
##                    self.SCALE = 1
##            except:
##                self.SCALE = 1
                
        if len(self.im)>0:
            self.Zoom_Canvas.Open_Image(image_pointer=self.im[sheet]) #,minW=cszw,minH=cszh)
        
        for seg in self.segID:
            pass
            #self.PreviewCanvas.delete(seg)
        self.segID = []

        for i in range(len(self.Notation_boxes)):
        #    print(self.Notation_boxes[i].coords)
            self.plot_polygon_new(self.Notation_boxes[i].coords)

    ################################################################################
    #                         General Settings Window                              #
    ################################################################################
    def GEN_Settings_Window(self):
        gen_settings = Toplevel(width=560, height=360)
        gen_settings.grab_set() # Use grab_set to prevent user input in the main window during calculations
        gen_settings.resizable(0,0)
        gen_settings.title('Settings')
        gen_settings.iconname("Settings")

        D_Yloc  = 6
        D_dY = 24
        xd_label_L = 12

        w_label=110
        w_entry=60
        w_units=35
        xd_entry_L=xd_label_L+w_label+10
        xd_units_L=xd_entry_L+w_entry+5

        #Radio Button

        D_Yloc=D_Yloc+D_dY
        self.Label_Resolution = Label(gen_settings,text="Resolution")
        self.Label_Resolution.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.Label_Resolution_u = Label(gen_settings,text="DPI", anchor=W)
        self.Label_Resolution_u.place(x=xd_units_L, y=D_Yloc, width=w_units, height=21)
        self.Entry_Resolution = Entry(gen_settings,width="15")
        self.Entry_Resolution.place(x=xd_entry_L, y=D_Yloc, width=w_entry, height=23)
        self.Entry_Resolution.configure(textvariable=self.resolution)
        trace_variable(self.resolution, self.Entry_Resolution_Callback)
        self.entry_set(self.Entry_Resolution,self.Entry_Resolution_Check(),2)

        ############################
        
        def get_Old_Color():
            color = askcolor("#%02x%02x%02x"%(self.OLD_COLOR[0:3])) 
            if color[0] == None:
                return
            self.OLD_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),255)
            self.Old_Color_Button.config(bg="#%02x%02x%02x"%(self.OLD_COLOR[0:3]))
            if self.im  != []:
                self.Reload_images()

        def get_New_Color():
            color = askcolor("#%02x%02x%02x"%(self.NEW_COLOR[0:3]))
            if color[0] == None:
                return
            self.NEW_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),255)
            self.New_Color_Button.config(bg="#%02x%02x%02x"%(self.NEW_COLOR[0:3]))
            if self.im  != []:
                self.Reload_images()

        def get_Highlight_Color():
            color = askcolor("#%02x%02x%02x"%(self.HIGHLIGHT_COLOR[0:3]))
            if color[0] == None:
                return
            self.HIGHLIGHT_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),255)
            self.Highlight_Color_Button.config(bg="#%02x%02x%02x"%(self.HIGHLIGHT_COLOR[0:3]))
            if self.im  != []:
                self.Reload_images()

        def get_Both_Color():
            color = askcolor("#%02x%02x%02x"%(self.BOTH_COLOR[0:3]))
            if color[0] == None:
                return
            self.BOTH_COLOR = (int(color[0][0]), int(color[0][1]),int(color[0][2]),255)
            self.Both_Color_Button.config(bg="#%02x%02x%02x"%(self.BOTH_COLOR[0:3]))
            if self.im  != []:
                self.Reload_images()
            
        D_Yloc=D_Yloc+D_dY+10
        self.Label_Old_Color = Label(gen_settings,text="Old Color")
        self.Label_Old_Color.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.Old_Color_Button = Button(gen_settings,text='', command=get_Old_Color)
        self.Old_Color_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        D_Yloc=D_Yloc+D_dY+10
        self.Label_New_Color = Label(gen_settings,text="New Color")
        self.Label_New_Color.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.New_Color_Button = Button(gen_settings,text='', command=get_New_Color)
        self.New_Color_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        D_Yloc=D_Yloc+D_dY+10
        self.Label_Highlight_Color = Label(gen_settings,text="Highlight Color")
        self.Label_Highlight_Color.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.Highlight_Color_Button = Button(gen_settings,text='', command=get_Highlight_Color)
        self.Highlight_Color_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        D_Yloc=D_Yloc+D_dY+10
        self.Label_Both_Color = Label(gen_settings,text="Both Color")
        self.Label_Both_Color.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.Both_Color_Button = Button(gen_settings,text='', command=get_Both_Color)
        self.Both_Color_Button.place(x=xd_entry_L, y=D_Yloc, width=50, height=21)

        self.Old_Color_Button.config(bg="#%02x%02x%02x"%(self.OLD_COLOR[0:3]))
        self.New_Color_Button.config(bg="#%02x%02x%02x"%(self.NEW_COLOR[0:3]))
        self.Highlight_Color_Button.config(bg="#%02x%02x%02x"%(self.HIGHLIGHT_COLOR[0:3]))
        self.Both_Color_Button.config(bg="#%02x%02x%02x"%(self.BOTH_COLOR[0:3]))

        D_Yloc=D_Yloc+D_dY+10
        self.Label_GS_path = Label(gen_settings,text="Ghostscript Path")
        self.Label_GS_path.place(x=xd_label_L, y=D_Yloc, width=w_label, height=21)
        self.Entry_GS_path = Entry(gen_settings,width="15")
        self.Entry_GS_path.place(x=xd_entry_L, y=D_Yloc, width=300, height=23)
        self.Entry_GS_path.configure(textvariable=self.gs_path)

        ############################
        D_Yloc=D_Yloc+D_dY+10
        self.Label_SaveConfig = Label(gen_settings,text="Configuration File")
        self.Label_SaveConfig.place(x=xd_label_L, y=D_Yloc, width=113, height=21)

        self.GEN_SaveConfig = Button(gen_settings,text="Save")
        self.GEN_SaveConfig.place(x=xd_entry_L, y=D_Yloc, width=w_entry, height=21, anchor="nw")
        self.GEN_SaveConfig.bind("<ButtonRelease-1>", self.Write_Config_File)

        
        ## Buttons ##
        gen_settings.update_idletasks()
        Ybut=int(gen_settings.winfo_height())-30
        Xbut=int(gen_settings.winfo_width()/2)

        self.GEN_Close = Button(gen_settings,text="Close",command=self.Close_Current_Window_Click)
        self.GEN_Close.place(x=Xbut, y=Ybut, width=130, height=30, anchor="center")

        
################################################################################
#                          Image Compare Operation                             #
################################################################################
def image_op_blur(PIL_old,PIL_new,HIGHLIGHT=False,    
    OLD_COLOR       = (255,255,255,255),
    NEW_COLOR       = (255,0  ,0  ,255),
    BOTH_COLOR      = (0  ,0  ,0  ,255),
    HIGHLIGHT_COLOR = (255,250,0  ,255),
    n = 10): #n: highlight distance
    
    if (PIL_old!=[] and PIL_new!=[]):
        wim_old, him_old = PIL_old.size
        wim_new, him_new = PIL_new.size
        wim = min(wim_old,wim_new)
        him = min(him_old,him_new)

        box = (0,0,wim,him)
        PIL_old = PIL_old.crop(box)
        PIL_new = PIL_new.crop(box)

    elif PIL_old!=[]:
        wim,him = PIL_old.size
        PIL_new = Image.new("L", (wim,him),(255))
        NEW_COLOR = (255,0  ,0  ,255)
        HIGHLIGHT = False
        
    elif PIL_new!=[]:
        wim,him = PIL_new.size
        PIL_old = Image.new("L", (wim,him),(255))
        OLD_COLOR = (0  ,0  ,255,255)
        HIGHLIGHT = False

    #if HIGHLIGHT==False:
    #    NEW_COLOR = (255,0  ,0  ,255)
    #    OLD_COLOR = (0  ,0  ,255,255)
    ###############################################################
    #print "Combining images."
    ###############################################################
    newL = PIL_new.convert("L")
    oldL = PIL_old.convert("L")
    #img_old_only = ImageMath.lambda_eval("convert( new & ~old,'L')", old=PIL_old,new=PIL_new)
    #img_new_only = ImageMath.lambda_eval("convert(~new &  old,'L')", old=PIL_old,new=PIL_new)
    img_both = ImageMath.unsafe_eval("convert( new |  old,'L')", old=PIL_old,new=PIL_new)
    combRGBA = Image.new("RGBA", (wim,him),(255,255,255,0))

    if HIGHLIGHT:
        ###############################################################
        #print "Highlighting image"
        ###############################################################
        
        # Highlight only old
        #img_highlight = ImageMath.lambda_eval("convert( new & ~old,'L')", old=PIL_old,new=PIL_new) 
        # Highlight new and old
        img_highlight = ImageMath.unsafe_eval("convert( new ^ old,'L')", old=PIL_old,new=PIL_new)
        img_highlight = img_highlight.filter(ImageFilter.GaussianBlur(radius=n))
        img_highlight = ImageMath.unsafe_eval("convert(a*1000,'L')", a=img_highlight)
        combRGBA.paste(HIGHLIGHT_COLOR,None,img_highlight)
    
    combRGBA.paste(OLD_COLOR ,None,ImageOps.invert(PIL_old))
    combRGBA.paste(NEW_COLOR ,None,ImageOps.invert(PIL_new))
    combRGBA.paste(BOTH_COLOR,None,ImageOps.invert(img_both))
    combRGBA = combRGBA.convert('RGB')
    ###############################################################
    return combRGBA

def Imaging_Free(image_in):
    b = io.BytesIO()
    image_in.save(b,'PPM')
    image_bytes = b.getvalue()
    image_out = PhotoImage(data=image_bytes, format='PPM')
    return image_out

#################################################################################
# Advanced zoom for images of various types from small to huge up to several GB #
#################################################################################
class AutoScrollbar(ttk.Scrollbar):
    """ A scrollbar that hides itself if it's not needed. Works only for grid geometry manager """
    def set(self, lo, hi):
        #if float(lo) <= 0.0 and float(hi) >= 1.0:
        #    self.grid_remove()
        #else:
        self.grid()
        ttk.Scrollbar.set(self, lo, hi)

    def pack(self, **kw):
        raise TclError('Cannot use pack with the widget ' + self.__class__.__name__)

    def place(self, **kw):
        raise TclError('Cannot use place with the widget ' + self.__class__.__name__)


class Notation_Box:
    def __init__(self, coords=[]):
        self.text  = ""
        self.coords = coords
        self.sheet = None
        self.zone  = None
        self.image = None
        self.calc_zone()

    def get_coords(self):
        return coords

    def set_coords(self,coords):
        self.coords=coords
        self.calc_zone()

    def calc_zone(self):
        self.zone=None
    


class CanvasImage:
    """ Display and zoom image """
    def __init__(self, placeholder, path):
        """ Initialize the ImageFrame """
        self.imscale = 1.0   # scale for the canvas image zoom, public for outer classes
        self.__delta = 1.3   # zoom magnitude
        self.MaxZoom = self.__delta**7 # Maximum Zoom 1.3*1.3*1.3 = 2.197
        self.__filter = Image.LANCZOS  # could be: NEAREST, BILINEAR, BICUBIC and LANCZOS
        self.__previous_state = 0  # previous state of the keyboard
        self.imwidth=1
        self.imheight=1
        self.imscale=1
        self.__image=None
        #self.path = path  # path to the image, should be public for outer classes
        # Create ImageFrame in placeholder widget
        self.__imframe = ttk.Frame(placeholder)  # placeholder of the ImageFrame object
        # Vertical and horizontal scrollbars for canvas
        hbar = AutoScrollbar(self.__imframe, orient='horizontal')
        vbar = AutoScrollbar(self.__imframe, orient='vertical')
        hbar.grid(row=1, column=0, sticky='we')
        vbar.grid(row=0, column=1, sticky='ns')
        # Create canvas and bind it with scrollbars. Public for outer classes
        self.canvas = Canvas(self.__imframe, highlightthickness=0, xscrollcommand=hbar.set, yscrollcommand=vbar.set)
        self.canvas.grid(row=0, column=0, sticky='nswe')
        self.canvas.update()  # wait till canvas is created
        hbar.configure(command=self.__scroll_x)  # bind scrollbars to the canvas
        vbar.configure(command=self.__scroll_y)
        # Bind events to the Canvas
        self.canvas.bind('<Configure>', lambda event: self.__show_image())  # canvas is resized
        self.canvas.bind('<ButtonPress-3>', self.__move_from)  # remember canvas position
        self.canvas.bind('<ButtonPress-2>', self.__move_from)  # remember canvas position
        self.canvas.bind('<B3-Motion>',     self.__move_to)  # move canvas to the new position
        self.canvas.bind('<B2-Motion>',     self.__move_to)  # move canvas to the new position
        self.canvas.bind('<MouseWheel>',    self.__wheel)  # zoom for Windows and MacOS, but not Linux
        self.canvas.bind('<Button-5>',      self.__wheel)  # zoom for Linux, wheel scroll down
        self.canvas.bind('<Button-4>',      self.__wheel)  # zoom for Linux, wheel scroll up
        # Handle keystrokes in idle mode, because program slows down on a weak computers,
        # when too many key stroke event in the same time
        self.canvas.bind('<Key>', lambda event: self.canvas.after_idle(self.__keystroke, event))

        self.__band_width = 1024  # width of the tile band
        if path!=None:
            self.Open_Image(path)

    def Open_Image(self,image_path=None,image_pointer=None):                    
        self.path = image_path  # path to the image, should be public for outer classes
        self.imscale = 1.0      # scale for the canvas image zoom, public for outer classes
        if image_pointer != None:
            self.full_image = image_pointer.copy()
            self.__image    = image_pointer
        else:
            self.__image = Image.open(self.path)  # open image
            
        self.imwidth, self.imheight = self.__image.size  # public for outer classes
        self.__min_height = self.imheight  # get the image height
        # Create image pyramid
        self.__pyramid = [self.full_image.copy()]
        # Set ratio coefficient for image pyramid
        self.__ratio = 1.0
        
        self.__curr_img = 0  # current image from the pyramid
        self.__scale = self.imscale * self.__ratio  # image pyramide scale
        self.__reduction = 2  # reduction degree of image pyramid
        w, h = self.__pyramid[-1].size
        while w > 512 and h > 512:  # top pyramid image is around 512 pixels in size
            w /= self.__reduction  # divide on reduction degree
            h /= self.__reduction  # divide on reduction degree
            self.__pyramid.append(self.__pyramid[-1].resize((int(w), int(h)), self.__filter))
        # Put image into container rectangle and use it to set proper coordinates to the image
        self.container = self.canvas.create_rectangle((0, 0, self.imwidth, self.imheight), width=0)
        ####

        #minW = int(self.canvas.cget("width"))
        #minH = int(self.canvas.cget("height"))
        #self.MinZoom = min( float(minW)/float(self.imwidth), float(minH)/float(self.imheight))
        #self.zoom_to(self.MinZoom)
        self.home_image()
        ####
        #self.__show_image()  # show image on the canvas
        self.canvas.focus_set()  # set focus on the canvas

    def redraw_figures(self):
        """ Dummy function to redraw figures in the children classes """
        pass

    def grid(self, **kw):
        """ Put CanvasImage widget on the parent widget """
        self.__imframe.grid(**kw)  # place CanvasImage widget on the grid
        self.__imframe.grid(sticky='nswe')  # make frame container sticky
        self.__imframe.rowconfigure(0, weight=1)  # make canvas expandable
        self.__imframe.columnconfigure(0, weight=1)

    def pack(self, **kw):
        """ Exception: cannot use pack with this widget """
        raise Exception('Cannot use pack with the widget ' + self.__class__.__name__)

    def place(self, **kw):
        """ Exception: cannot use place with this widget """
        raise Exception('Cannot use place with the widget ' + self.__class__.__name__)

    def home_image(self):
        if self.__image==None:
            return
        minW = int(self.canvas.cget("width"))
        minH = int(self.canvas.cget("height"))
        self.MinZoom = min( float(minW)/float(self.imwidth), float(minH)/float(self.imheight))
        self.zoom_to(self.MinZoom)        
        self.canvas.xview_scroll(1000, "unit")
        self.canvas.yview_scroll(1000, "unit")
        self.__show_image()   # redraw the image
        self.canvas.xview_scroll(-1000, "unit")
        self.canvas.yview_scroll(-1000, "unit")
        self.__show_image()   # redraw the image
        self.canvas.xview_moveto(0)
        self.canvas.yview_moveto(0)
        self.__show_image()   # redraw the image
        
    # noinspection PyUnusedLocal
    def __scroll_x(self, *args, **kwargs):
        """ Scroll canvas horizontally and redraw the image """
        self.canvas.xview(*args)  # scroll horizontally
        self.__show_image()  # redraw the image

    # noinspection PyUnusedLocal
    def __scroll_y(self, *args, **kwargs):
        """ Scroll canvas vertically and redraw the image """
        self.canvas.yview(*args)  # scroll vertically
        self.__show_image()  # redraw the image

    def __show_image(self):
        #print("001 SCALE: %f" %(self.imscale))
        """ Show image on the Canvas. Implements correct image zoom almost like in Google Maps """
        try:
            box_image = self.canvas.coords(self.container)  # get image area
        except:
            return
        box_canvas = (self.canvas.canvasx(0),  # get visible area of the canvas
                      self.canvas.canvasy(0),
                      self.canvas.canvasx(self.canvas.winfo_width()),
                      self.canvas.canvasy(self.canvas.winfo_height()))
        box_img_int = tuple(map(int, box_image))  # convert to integer or it will not work properly
        if box_img_int==():
            return
        # Get scroll region box
        box_scroll = [min(box_img_int[0], box_canvas[0]), min(box_img_int[1], box_canvas[1]),
                      max(box_img_int[2], box_canvas[2]), max(box_img_int[3], box_canvas[3])]
        # Horizontal part of the image is in the visible area
        if  box_scroll[0] == box_canvas[0] and box_scroll[2] == box_canvas[2]:
            box_scroll[0]  = box_img_int[0]
            box_scroll[2]  = box_img_int[2]
        # Vertical part of the image is in the visible area
        if  box_scroll[1] == box_canvas[1] and box_scroll[3] == box_canvas[3]:
            box_scroll[1]  = box_img_int[1]
            box_scroll[3]  = box_img_int[3]
        # Convert scroll region to tuple and to integer
        self.canvas.configure(scrollregion=tuple(map(int, box_scroll)))  # set scroll region
        x1 = max(box_canvas[0] - box_image[0], 0)  # get coordinates (x1,y1,x2,y2) of the image tile
        y1 = max(box_canvas[1] - box_image[1], 0)
        x2 = min(box_canvas[2], box_image[2]) - box_image[0]
        y2 = min(box_canvas[3], box_image[3]) - box_image[1]

        if self.__scale>1.0:
            IM_filter = Image.NEAREST # could be: NEAREST, BILINEAR, BICUBIC and LANCZOS 
        else:
            IM_filter = Image.LANCZOS
            
        if int(x2 - x1) > 0 and int(y2 - y1) > 0:  # show image if it in the visible area
            image = self.__pyramid[max(0, self.__curr_img)].crop(  # crop current img from pyramid
                                  (int(x1 / self.__scale), int(y1 / self.__scale),
                                   int(x2 / self.__scale), int(y2 / self.__scale)))
            try:
                imagetk = ImageTk.PhotoImage(image.resize((int(x2 - x1), int(y2 - y1)), IM_filter))
            except:
                imagetk = Imaging_Free(image.resize((int(x2 - x1), int(y2 - y1)), IM_filter))
                
            self.image_location=[ max(box_canvas[0], box_img_int[0]) , max(box_canvas[1], box_img_int[1]),int(x1),int(y1),self.imscale]
            imageid = self.canvas.create_image(self.image_location[0], self.image_location[1], anchor='nw', image=imagetk)
            #imageid = self.canvas.create_image(max(box_canvas[0], box_img_int[0]),
            #                                   max(box_canvas[1], box_img_int[1]),
            #                                   anchor='nw', image=imagetk)

            self.canvas.lower(imageid)  # set image into background
            self.canvas.imagetk = imagetk  # keep an extra reference to prevent garbage-collection

    def __move_from(self, event):
        """ Remember previous coordinates for scrolling with the mouse """
        self.canvas.scan_mark(event.x, event.y)

    def __move_to(self, event):
        """ Drag (move) canvas to the new position """
        self.canvas.scan_dragto(event.x, event.y, gain=1)
        self.__show_image()  # zoom tile and show it on the canvas

    def outside(self, x, y):
        """ Checks if the point (x,y) is outside the image area """
        bbox = self.canvas.coords(self.container)  # get image area
        if bbox==[]:
            return True
        if bbox[0] < x < bbox[2] and bbox[1] < y < bbox[3]:
            return False  # point (x,y) is inside the image area
        else:
            return True  # point (x,y) is outside the image area

    def __wheel(self, event):
        """ Zoom with mouse wheel """
        x = self.canvas.canvasx(event.x)  # get coordinates of the event on the canvas
        y = self.canvas.canvasy(event.y)
        if self.outside(x, y): return  # zoom only inside image area
        scale = 1.0
        # Respond to Linux (event.num) or Windows (event.delta) wheel event
        if event.num == 5 or event.delta == -120:  # scroll down, smaller
            if self.imscale == self.MinZoom:
                return
            if (self.imscale/self.__delta < self.MinZoom):
                scale        = self.MinZoom/self.imscale #self.MinZoom/self.imscale
                self.imscale = self.MinZoom
            else:
                self.imscale /= self.__delta
                scale        /= self.__delta
                
        if event.num == 4 or event.delta == 120:  # scroll up, bigger            
            if self.imscale == self.MaxZoom:
                return
            if self.imscale*self.__delta > self.MaxZoom:
                scale        = self.MaxZoom/self.imscale #self.MaxZoom/self.imscale
                self.imscale = self.MaxZoom
            else:
                self.imscale *= self.__delta
                scale        *= self.__delta
                
        # Take appropriate image from the pyramid
        k = self.imscale * self.__ratio  # temporary coefficient
        self.__curr_img = min((-1) * int(math.log(k, self.__reduction)), len(self.__pyramid) - 1)
        self.__scale = k * math.pow(self.__reduction, max(0, self.__curr_img))
        #
        self.canvas.scale('all', x, y, scale, scale)  # rescale all objects
        # Redraw some figures before showing image on the screen
        self.redraw_figures()  # method for child classes
        self.__show_image()

    def zoom_to(self, scale_in):
        if scale_in<=0:
            print("SCALE (%f) IS LESS THAN OR EQUAL TO 0.0" %(scale_in))
            scale_in=1
        scale = float(scale_in)/self.imscale
        self.imscale = float(scale_in)
        x=0
        y=0
        # Take appropriate image from the pyramid
        k = self.imscale * self.__ratio  # temporary coefficient
        self.__curr_img = min((-1) * int(math.log(k, self.__reduction)), len(self.__pyramid) - 1)
        self.__scale = k * math.pow(self.__reduction, max(0, self.__curr_img))
        #
        self.canvas.scale('all', x, y, scale, scale)  # rescale all objects
        # Redraw some figures before showing image on the screen
        self.redraw_figures()  # method for child classes
        self.__show_image()

    def __keystroke(self, event):
        """ Scrolling with the keyboard.
            Independent from the language of the keyboard, CapsLock, <Ctrl>+<key>, etc. """
        if event.state - self.__previous_state == 4:  # means that the Control key is pressed
            pass  # do nothing if Control key is pressed
        else:
            self.__previous_state = event.state  # remember the last keystroke state
            # Up, Down, Left, Right keystrokes
            if event.keycode in [68, 39, 102]:  # scroll right, keys 'd' or 'Right'
                self.__scroll_x('scroll',  1, 'unit', event=event)
            elif event.keycode in [65, 37, 100]:  # scroll left, keys 'a' or 'Left'
                self.__scroll_x('scroll', -1, 'unit', event=event)
            elif event.keycode in [87, 38, 104]:  # scroll up, keys 'w' or 'Up'
                self.__scroll_y('scroll', -1, 'unit', event=event)
            elif event.keycode in [83, 40, 98]:  # scroll down, keys 's' or 'Down'
                self.__scroll_y('scroll',  1, 'unit', event=event)

    def crop(self, bbox):
        """ Crop rectangle from the image and return it """
        return self.__pyramid[0].crop(bbox)

    def destroy(self):
        """ ImageFrame destructor """
        self.__image.close()
        map(lambda i: i.close, self.__pyramid)  # close all pyramid images
        del self.__pyramid[:]  # delete pyramid list
        del self.__pyramid     # delete pyramid variable
        self.canvas.destroy()
        self.__imframe.destroy()
        self.__image=None


################################################################################
#                      Subprocess timout stuff                                 #
################################################################################
from subprocess import Popen, PIPE, STARTUPINFO, STARTF_USESHOWWINDOW
from threading import Timer
def run_external(cmd, timeout_sec):
    FLAG=[True]
    try:
        #print(cmd)
        startupinfo=None
        if sys.platform == 'win32':
            # This startupinfo structure prevents a console window from popping up on Windows
            startupinfo = STARTUPINFO()
            startupinfo.dwFlags |= STARTF_USESHOWWINDOW
            
        proc = Popen(cmd, shell=False, stdout=PIPE, stderr=PIPE, stdin=PIPE, startupinfo=startupinfo)

    except Exception as e:
        raise Exception("\n%s\n\nExecutable Path:\n%s" %(e,cmd[0]))
    kill_proc = lambda p: kill_sub_process(p,timeout_sec, FLAG)
    timer = Timer(timeout_sec, kill_proc, [proc])
    try:
        timer.start()
        stdout,stderr = proc.communicate()
    finally:
        timer.cancel()
    if not FLAG[0]:
        raise Exception("\nsub-process terminated after %d seconds." %(timeout_sec))
    return stdout,stderr
        
def kill_sub_process(p,timeout_sec, FLAG):
    FLAG[0]=False
    p.kill()
    
################################################################################
#                         Debug Message Box                                    #
################################################################################
def debug_message(message):
    global DEBUG
    title = "Debug Message"
    if DEBUG:
        if VERSION == 3:
            tkinter.messagebox.showinfo(title,message)
        else:
            tkMessageBox.showinfo(title,message)
            pass
        
################################################################################
#             Function for outputting messages to different locations          #
#            depending on what options are enabled                             #
################################################################################
def fmessage(text,newline=True):
    global QUIET
    end=''
    if (not QUIET):
        if newline==True:
            end='\n'
        if main_is_frozen():
            message_box("stdout:",text)
        else:
            sys.stdout.write(text+end)

################################################################################
#                               Message Box                                    #
################################################################################
def message_box(title,message):
    if VERSION == 3:
        tkinter.messagebox.showinfo(title,message)
    else:
        tkMessageBox.showinfo(title,message)
        pass

################################################################################
#                          Message Box ask OK/Cancel                           #
################################################################################
def message_ask_ok_cancel(title, mess):
    if VERSION == 3:
        result=tkinter.messagebox.askokcancel(title, mess)
    else:
        result=tkMessageBox.askokcancel(title, mess)
    return result

def main_is_frozen():
   return (hasattr(sys, "frozen") or # new py2exe or pyinstaller
           hasattr(sys, "importers")) # old py2exe

################################################################################
#                          Startup Application                                 #
################################################################################   
root = tkinterdnd2.Tk()
app = Application(root)
app.master.title(title_text)
app.master.iconname("DrawingDIFF")
app.master.minsize(780,540)

################################## Set Icon  ########################################
Icon_Set=False
if main_is_frozen():
    #print("frozen")
    try:
        root.iconbitmap(default=sys.argv[0])
        Icon_Set=True
    except:
        Icon_Set=False
        
if not Icon_Set:
    try:
        scorch_ico_B64=b'R0lGODlhEAAQAIYAAA\
        AAABAQEBYWFhcXFxsbGyUlJSYmJikpKSwsLC4uLi8vLzExMTMzMzc3Nzg4ODk5OTs7Oz4+PkJCQkRERE\
        VFRUtLS0xMTE5OTlNTU1dXV1xcXGBgYGVlZWhoaGtra3FxcXR0dHh4eICAgISEhI+Pj5mZmZ2dnaKioq\
        Ojo62tra6urrS0tLi4uLm5ub29vcLCwsbGxsjIyMzMzM/Pz9PT09XV1dbW1tjY2Nzc3OHh4eLi4uXl5e\
        fn5+jo6Ovr6+/v7/Hx8fLy8vT09PX19fn5+fv7+/z8/P7+/v///wAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\
        AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACH5BAEKAEkALAAAAAAQABAAQAj/AJMIFBhBQYAACRIkWbgwAA\
        4kEFEECACAxBAkGH8ESEKgBZIiAIQECBAjAA8kNwIkScKgQhAkRggAIJACCZIaJxgk2clgAY4OAAoEAO\
        ABCIIDSZIwkIHEBw0YFAAA6IGDCBIkLAhMyICka9cAKZCIRTLEBIMkaA0MSNGjSBEVIgpESEK3LgMCI1\
        aAWCFDA4EDSQInwaDACBEAImLwCAFARw4HFJJcgGADyZEAL3YQcMGBBpIjHx4EeIGkRoMFJgakWADABx\
        IkPwIgcIGkdm0AMJDo1g3jQBIBRZAINyKAwxEkyHEUSMIcwYYbEgwYmQGgyI8SD5Jo327hgIIAAQ5cBs\
        CQpHySgAA7'
        icon_im =PhotoImage(data=scorch_ico_B64, format='gif')
        root.call('wm', 'iconphoto', root._w, '-default', icon_im)
    except:
        pass
#####################################################################################

debug_message("Debugging is turned on.")
root.mainloop()
